<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;
use Gregwar\Captcha\CaptchaBuilder;

class AuthController extends Controller
{
    public function showLogin()
    {
        if (Auth::check()) {
            return $this->redirectBasedOnRole();
        }

        return view('auth.login');
    }

    public function generateCaptcha()
    {
        $builder = new CaptchaBuilder;
        $builder->build();

        Session::put('captcha', $builder->getPhrase());

        return response($builder->output(), 200, [
            'Content-Type' => 'image/jpeg',
        ]);
    }

    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
            'captcha' => 'required',
        ]);

        if (strtolower($request->captcha) !== strtolower(Session::get('captcha'))) {
            return back()->withErrors([
                'captcha' => 'Kode CAPTCHA salah',
            ])->withInput($request->except('password', 'captcha'));
        }

        $user = User::where('email', $request->email)->first();

        if (!$user) {
            return back()->withErrors([
                'email' => 'Email tidak terdaftar',
            ])->withInput($request->except('password', 'captcha'));
        }

        if ($user->status !== 'aktif') {
            return back()->withErrors([
                'email' => 'Akun Anda tidak aktif. Hubungi administrator.',
            ])->withInput($request->except('password', 'captcha'));
        }

        if (!Hash::check($request->password, $user->password)) {
            return back()->withErrors([
                'password' => 'Password salah',
            ])->withInput($request->except('password', 'captcha'));
        }

        Auth::login($user);
        $request->session()->regenerate();

        // ✅ Catat activity log setelah login berhasil
        activity_log(
            'auth',
            'login',
            'Login ke sistem dengan email: ' . $user->email,
            $user->id,
            $user->role
        );

        // ✅ PERBAIKAN: Cek dual access untuk SEMUA role admin (termasuk manager, gm, hrd)
        $hasAdminRole = in_array($user->role, ['admin', 'super_admin', 'manager', 'gm', 'hrd']);
        $hasKaryawanData = $user->karyawan && $user->karyawan->status === 'aktif';

        if ($hasAdminRole && $hasKaryawanData) {
            // User punya akses ganda, arahkan ke halaman pilihan role
            return redirect()->route('role.select');
        }

        // User hanya punya 1 role, redirect langsung
        return $this->redirectBasedOnRole();
    }

    /**
     * Tampilkan halaman pemilihan role
     */
    public function showRoleSelection()
    {
        $user = Auth::user();

        // ✅ PERBAIKAN: Validasi untuk SEMUA role admin
        $hasAdminRole = in_array($user->role, ['admin', 'super_admin', 'manager', 'gm', 'hrd']);
        $hasKaryawanData = $user->karyawan && $user->karyawan->status === 'aktif';

        if (!$hasAdminRole || !$hasKaryawanData) {
            return $this->redirectBasedOnRole();
        }

        return view('auth.role-select', compact('user'));
    }

    /**
     * Set role yang dipilih dan redirect
     */
    public function setRole(Request $request)
    {
        $request->validate([
            'selected_role' => 'required|in:admin,karyawan'
        ]);

        $user = Auth::user();

        // ✅ PERBAIKAN: Validasi dual access untuk SEMUA role admin
        $hasAdminRole = in_array($user->role, ['admin', 'super_admin', 'manager', 'gm', 'hrd']);
        $hasKaryawanData = $user->karyawan && $user->karyawan->status === 'aktif';

        if (!$hasAdminRole || !$hasKaryawanData) {
            return redirect()->route('login');
        }

        // Simpan dengan key 'active_role'
        Session::put('active_role', $request->selected_role);

        // ✅ Catat activity log pemilihan mode
        activity_log(
            'auth',
            'select_mode',
            'Memilih mode: ' . ucfirst($request->selected_role)
        );

        // Redirect berdasarkan role yang dipilih
        if ($request->selected_role === 'admin') {
            Session::put('original_role', $user->role);
            return redirect()->route('admin.dashboard');
        } else {
            return redirect()->route('karyawan.dashboard');
        }
    }

    /**
     * Switch role (untuk user yang sudah login)
     */
    public function switchRole(Request $request)
    {
        $request->validate([
            'switch_to' => 'required|in:admin,karyawan'
        ]);

        $user = Auth::user();

        // ✅ PERBAIKAN: Validasi dual access untuk SEMUA role admin
        $hasAdminRole = in_array($user->role, ['admin', 'super_admin', 'manager', 'gm', 'hrd']);
        $hasKaryawanData = $user->karyawan && $user->karyawan->status === 'aktif';

        if (!$hasAdminRole || !$hasKaryawanData) {
            return back()->with('error', 'Anda tidak memiliki akses ganda.');
        }

        // Update dengan key 'active_role'
        Session::put('active_role', $request->switch_to);

        // ✅ Catat activity log switch mode
        activity_log(
            'auth',
            'switch_mode',
            'Beralih ke mode ' . ucfirst($request->switch_to)
        );

        if ($request->switch_to === 'admin') {
            Session::put('original_role', $user->role);
            return redirect()->route('admin.dashboard')->with('success', 'Berhasil beralih ke mode Admin');
        } else {
            return redirect()->route('karyawan.dashboard')->with('success', 'Berhasil beralih ke mode Karyawan');
        }
    }

    public function logout(Request $request)
    {
        // ✅ Catat activity log SEBELUM logout
        if (Auth::check()) {
            $activeMode = Session::get('active_role', 'default');
            activity_log(
                'auth',
                'logout',
                'Logout dari sistem (mode: ' . $activeMode . ')'
            );
        }

        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('login');
    }

    private function redirectBasedOnRole()
    {
        $user = Auth::user();

        // Cek dengan key 'active_role'
        if (Session::has('active_role')) {
            $activeRole = Session::get('active_role');

            if ($activeRole === 'admin') {
                return redirect()->route('admin.dashboard');
            } else {
                return redirect()->route('karyawan.dashboard');
            }
        }

        // ✅ PERBAIKAN: Redirect untuk SEMUA role admin
        if (in_array($user->role, ['admin', 'super_admin', 'manager', 'gm', 'hrd'])) {
            return redirect()->route('admin.dashboard');
        }

        return redirect()->route('karyawan.dashboard');
    }
}
